import { appNoBuildInProgressOption, AppOtpFeedStatuses, AppOtpFeedStatusesResponse } from "@/zod/feeds.schema";
import { AppTransformedAgencySchema } from "@/zod/transforms.schema";

export type GeneratorOptions = undefined | null | Partial<{
  length: number
}>

function INTERNAL__optionsGuard(opts: GeneratorOptions) {
  return typeof opts?.length === 'number' && opts.length === 0;
}


/** Transformers */

/**
 * Generic transformer to map results based on transformer action.
 * 
 * 'generateAfter' is ran on all elements.
 * @param before 
 * @param generateAfter 
 * @returns TAfter
 */
export function transform<TBefore, TAfter>(
  before: TBefore[],
  generateAfter: ((before: TBefore) => TAfter),
  opts?: GeneratorOptions,
) {
  if (!Array.isArray(before) || INTERNAL__optionsGuard(opts)) {
    return null;
  }

  return Array.from(INTERNAL__transformGenerator(before, generateAfter, opts))
}

/**
 * Generic transformer to map nested results based on transformer actions.
 * 
 * 'generateAfter' is ran on all nested elements.
 * @param before 
 * @param generateIntermediate 
 * @returns TAfter
 */
export function transformNested<TBefore, TIntermediate, TAfter>(
  before: TBefore[],
  generateIntermediate: ((before: TBefore) => TIntermediate[]),
  generateAfter: ((before: TBefore, intermediate: TIntermediate) => TAfter)
) {
  if (!Array.isArray(before)) {
    return null;
  }

  return Array.from(INTERNAL__transformNestedGenerator(before, generateIntermediate, generateAfter))
}


/** Generators */
function* INTERNAL__transformGenerator<TBefore, TAfter>(
  before: TBefore[],
  generateAfter: ((before: TBefore) => TAfter),
  opts?: GeneratorOptions,
) {
  let length = 0;

  for (const b of before) {
    if (typeof opts?.length === 'number' && length >= opts.length) {
      break;
    }

    length++;
    yield generateAfter(b)
  }
}


function* INTERNAL__transformNestedGenerator<TBefore, TIntermediate, TAfter>(
  before: TBefore[],
  generateIntermediate: ((before: TBefore) => TIntermediate[]),
  generateAfter: ((before: TBefore, intermediate: TIntermediate) => TAfter),
) {
  for (const b of before) {
    const intermediate = generateIntermediate(b)
    for (const i of intermediate) {
      yield generateAfter(b, i)
    }
  }
}

export function transformAgenciesResponse(services: AppOtpFeedStatusesResponse) {
  if (!services?.body) {
    throw new Error(`transformAgenciesResponse, Unable to transform undefined ${services}`)
  }

  return transformAgenciesData(services.body)
}

export function transformAgenciesData(agencies: AppOtpFeedStatuses) {
  return transform(
    agencies,
    (agency) => ({
      access: agency.agency_access ?? "UNKNOWN",
      id: agency.id!,
      gtfsId: agency.gtfsId!,
      name: agency.name!,
      url: agency.url!,
      phone: agency.phone!,
      path: agency.path!,
      error: {
        message: agency.validation_status.error,
      },
      status: {
        started: agency.validation_status.startDate ? new Date(agency.validation_status.startDate) : undefined,
        option: agency.validation_status.status ?? appNoBuildInProgressOption,
      },
      dates: {
        uploaded: agency.date_upload ? new Date(agency.date_upload) : undefined,
        uploadedToPreview: agency.date_preview ? new Date(agency.date_preview) : undefined,
        uploadedToLive: agency.date_live ? new Date(agency.date_live) : undefined,
      }
    } satisfies AppTransformedAgencySchema)
  )
}

